<?php

namespace App\Controllers;

use App\Libraries\CIAuth;

use App\Models\CIAuthModel;
use App\Models\PanelModel;
use App\Models\HomeSectionsModel;
use App\Models\CategoryModel;
use App\Models\CountriesModel;
use App\Models\RadioModel;
use App\Models\PodcastModel;

class SectionsController extends BaseController {
    
    public function __construct(){
        $panelModel = new PanelModel();
        $this->settings = $panelModel->getSettings();
        $this->details = new HomeSectionsModel();
    }

    public function index(){
        if(!CIAuth::check()){
            return redirect()->to(base_url('ns-admin/login'));
        }
        
        $data = [
            'pageTitle' => 'Manage Sections',
            'currentFile' => 'sections'
        ];
        
        $data['settings'] = $this->settings;
        
        $keyword = $this->request->getGet('keyword') ?? '';
        if (!empty($keyword)) {
            $data['result'] = $this->details
                ->orderBy('id', 'DESC')
                ->like('section_name', $keyword)
                ->findAll();
        } else {
            $data['result'] = $this->details->orderBy('id', 'DESC')->findAll();
        }
        return view('manage_sections', $data);
    }
    
    public function create(){
        if(!CIAuth::check()){
            return redirect()->to(base_url('ns-admin/login'));
        }
        
        $data = [
            'pageTitle' => 'Create Sections',
            'currentFile' => 'sections',
            'pageSave' => 'Create'
        ];
        
        $category = new CategoryModel();
        $data['resultCategory'] = $category->orderBy('cid', 'DESC')->findAll();
        
        $countries = new CountriesModel();
        $data['resultCountries'] = $countries->orderBy('sid', 'DESC')->findAll();
        
        $radios = new RadioModel();
        $data['resultRadios'] = $radios->orderBy('id', 'DESC')->findAll();
        
        $podcast = new PodcastModel();
        $data['resultPodcast'] = $podcast->orderBy('pid', 'DESC')->findAll();
        
        $data['settings'] = $this->settings;
        return view('create_sections', $data);
    }
    
    public function edit($id = null){
        if(!CIAuth::check()){
            return redirect()->to(base_url('ns-admin/login'));
        }
        
        $dataDetails = $this->details->find($id);
        
        $data = [
            'pageTitle' => 'Edit Sections',
            'currentFile' => 'sections',
            'pageSave' => 'Save',
            'section_id' => $dataDetails['id'],
            'row' => $dataDetails,
        ];
        
        $category = new CategoryModel();
        $data['resultCategory'] = $category->orderBy('cid', 'DESC')->findAll();
        
        $countries = new CountriesModel();
        $data['resultCountries'] = $countries->orderBy('sid', 'DESC')->findAll();
        
        $radios = new RadioModel();
        $data['resultRadios'] = $radios->orderBy('id', 'DESC')->findAll();
        
        $podcast = new PodcastModel();
        $data['resultPodcast'] = $podcast->orderBy('pid', 'DESC')->findAll();
        
        $data['settings'] = $this->settings;
        return view('create_sections', $data);
    }
    
    public function createHandler(){
        if(!CIAuth::check()){
            return redirect()->to(base_url('ns-admin/login'));
        }
        
        $editID = $this->request->getPost('section_id');
        
        // Simple CSRF check with better error handling
        $postedToken = $this->request->getPost('csrf_test_name');
        if (empty($postedToken)) {
            $message = array('message' => 'CSRF token is missing. Please refresh the page and try again.', 'class' => 'error');
            session()->set('response_msg', $message);
            if(empty($editID)){
                return redirect()->to(base_url('ns-admin/manage-sections'));
            } else {
              return redirect()->to(base_url('ns-admin/create-sections/'.$editID));  
            }
        }
        if ($postedToken !== csrf_hash()) {
            $message = array('message' => 'Security token validation failed. Please refresh the page and try again.', 'class' => 'error');
            session()->set('response_msg', $message);
            if(empty($editID)){
                return redirect()->to(base_url('ns-admin/manage-sections'));
            } else {
              return redirect()->to(base_url('ns-admin/create-sections/'.$editID));  
            }
        }

        if(empty($editID)){
            
            $section_type = trim($this->request->getPost('section_type'));
            
            if($section_type=='category'){
                $post_ids = implode(',',$this->request->getPost('cat_post_id'));
            } else if($section_type=='countries'){
                $post_ids = implode(',',$this->request->getPost('countries_post_id'));
            } else if($section_type=='radio'){
                $post_ids = implode(',',$this->request->getPost('radio_post_id'));
            } else if($section_type=='podcasts'){
                $post_ids = implode(',',$this->request->getPost('podcast_post_id'));
            } else if($section_type=='banner_ads'){
                $post_ids = "";
            }
            
            $data = array( 
                'section_name'  =>  $this->request->getPost('section_name'),
                'section_type'  =>  $section_type,
                'post_ids'  =>  $post_ids
            );
            
            $this->details->insert($data);
            
            $message = array('message' => lang('Validation.added'),'class' => 'success');
            session()->set('response_msg', $message);
            return redirect()->to(base_url('ns-admin/manage-sections'));
            
        } else {
            
            $dataDetails = $this->details->find($editID);
            if($dataDetails){
                if ($this->request->getPost('submit') !== null) {
                    $section_type = trim($this->request->getPost('section_type'));
                    
                    if($section_type=='category'){
                        $post_ids = implode(',',$this->request->getPost('cat_post_id'));
                    } else if($section_type=='countries'){
                        $post_ids = implode(',',$this->request->getPost('countries_post_id'));
                    } else if($section_type=='radio'){
                        $post_ids = implode(',',$this->request->getPost('radio_post_id'));
                    } else if($section_type=='podcasts'){
                        $post_ids = implode(',',$this->request->getPost('podcast_post_id'));
                    } else if($section_type=='banner_ads'){
                        $post_ids = "";
                    }
                    
                    $data = array( 
                        'section_name'  =>  $this->request->getPost('section_name'),
                        'section_type'  =>  $section_type,
                        'post_ids'  =>  $post_ids
                    );
                    $this->details->update($editID, $data);
                    $message = array('message' => lang('Validation.updated'),'class' => 'success');
                } else {
                    $message = array('message' => lang('Validation.updated_failed'),'class' => 'error');
                }
            } else {
                $message = array('message' => lang('Validation.updated_failed_user'),'class' => 'error');
            }
            session()->set('response_msg', $message);
            return redirect()->to(base_url('ns-admin/create-sections/'.$editID));
        }
    }
    
    public function delete($id = null){
        // Check if the user is authorized to perform the action
        if (!CIAuth::check()) {
            return $this->response->setJSON(['status' => 0, 'message' => 'Unauthorized']);
        }
        
        $allowedAdminTypes = [1, 3];
        if(!in_array(CIAuth::adminType(), $allowedAdminTypes)){
            return $this->response->setJSON(['status' => 0, 'message' => 'Unauthorized']);
        }
        
        // Validate ID
        if (is_null($id) || !is_numeric($id)) {
            return $this->response->setJSON(['status' => 0, 'message' => 'Invalid ID']);
        }
        
        // Check if the record exists
        $record = $this->details->find($id);
        if (!$record) {
            return $this->response->setJSON(['status' => 0, 'message' => 'Record not found']);
        }
        
        try {
            // Attempt to delete the record
            if ($this->details->delete($id)) {
                session()->set('response_msg', ['message' => lang('Validation.deleted'), 'class' => 'success']);
                return $this->response->setJSON(['status' => 1, 'message' => 'Delete successful']);
            } else {
                return $this->response->setJSON(['status' => 0, 'message' => 'Failed to delete']);
            }
        } catch (\Exception $e) {
            // Log and handle potential errors during deletion
            log_message('error', 'Error deleting record: ' . $e->getMessage());
            return $this->response->setJSON(['status' => 0, 'message' => 'An error occurred while deleting']);
        }
    }
    
    public function status($id = null){
        // Ensure user is authenticated
        if (!CIAuth::check()) {
            return $this->response->setStatusCode(401)
                ->setJSON(['status' => 0, 'msg' => 'Unauthorized']);
        }
        
        $allowedAdminTypes = [1, 3];
        if(!in_array(CIAuth::adminType(), $allowedAdminTypes)){
            return $this->response->setStatusCode(401)
                ->setJSON(['status' => 0, 'msg' => 'Unauthorized']);
        }
        
        // Validate ID
        if (is_null($id) || !is_numeric($id)) {
            return $this->response->setJSON(['status' => 0, 'msg' => 'Invalid ID']);
        }
        
        // Check if record exists
        $record = $this->details->find($id);
        if (!$record) {
            return $this->response->setJSON(['status' => 0, 'msg' => 'Record not found']);
        }
        
        // Ensure the request is a POST request
        if (!$this->request->is('post')) {
            return $this->response->setStatusCode(405)
                ->setJSON(['status' => 0, 'msg' => 'Method Not Allowed']);
        }
        
        try {
            $for_action = $this->request->getPost('for_action');
            $column = $this->request->getPost('column');
            
            // Update data
            $data = [$column => ($for_action === 'enable' ? '1' : '0')];
            $message = ($for_action === 'enable') ? lang('Validation.enabled') : lang('Validation.disabled');
            
            $this->details->update($id, $data);
            
            return $this->response->setJSON([
                'status' => 1,
                'action' => $for_action,
                'msg' => $message,
                'class' => 'success'
            ]);
        } catch (\Exception $e) {
            return $this->response->setJSON([
                'status' => 0,
                'msg' => 'An error occurred while updating the record'
            ]);
        }
    }
}
